<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\ForgotPasswordRequest;
use App\Http\Requests\LoginRequest;
use App\Http\Requests\SignupRequest;
use App\Http\Requests\UpdatePasswordRequest;
use App\Http\Requests\UpdateProfileRequest;
use App\Http\Requests\VerifyOtpRequest;
use App\Library\StripePaymentGateway;
use App\Mail\ForgotPassword;
use App\Mail\RegisterOtp;
use App\Models\AdvisorCategory;
use App\Models\AdvisorServicePrice;
use App\Models\BookingRequest;
use App\Models\CountryCode;
use App\Models\FavoriteAdvisor;
use App\Models\LoginLog;
use App\Models\Notification;
use App\Models\User;
use Artisan;
use Auth;
use Hash;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Http\Request;
use DB;
use Exception;
use Illuminate\Support\Facades\Log;
use Mail;
use Response;
use Session;
use Validator;
use App\Traits\OrderTrait;

use Illuminate\Support\Facades\Http;

class UserController extends Controller
{
  public function signup(SignupRequest $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      DB::beginTransaction();

      $requestData = $request->all();

      // CHECK EMAIL ALREADY EXIST
      $emailCheck = User::where('email', $requestData['email'])
        ->first();

      if ($emailCheck) {
        return error_response('The email has already been taken.', STATUS_UNPROCESSABLE);
      }

      $requestData['password'] = bcrypt($requestData['password']);

      $userObj = User::create($requestData);

      if ($request->hasFile('image')) {
        if (!is_dir(IMAGE_UPLOAD_PATH)) {
          mkdir(IMAGE_UPLOAD_PATH, 0777, true);
        }

        $file = $request->file('image');
        $fileName = time() . '-' . $file->getClientOriginalName();
        $file->move(IMAGE_UPLOAD_PATH, $fileName);
        $userObj->image = $fileName;

        $userObj->save();
      }

      // GENERATE OTP
      $otp = generate_otp(6);
      $userObj->otp = $otp;
      $userObj->otp_expired_at = date("Y-m-d H:i:s", strtotime("+10 minutes"));
      $userObj->save();

      // SAVE 3 FREE MINUTES IF USER TYPE IS 2 (CUSTOMER)
      if ($userObj->user_type == 2) {
        $userObj->free_minutes = "3:00";
        $userObj->save();
      }

      $response['message'] = 'User registerd successfully';

      $token = $userObj->createToken($userObj->id . ' token ')->accessToken;

      User::where('id', $userObj->id)->update([
        'device_type' => $requestData['device_type'] ?? NULL,
        'device_token' => $requestData['device_token'] ?? NULL,
        'access_token' => $token,
      ]);

      if ($token) {
        DB::commit();

        // CREATE STRIPE CUSTOMER ID
        if ($userObj->user_type == 2) {
          create_save_stripe_customer_id(StripePaymentGateway::class, $userObj, [
            'email' => $userObj->email
          ]);
        }
      }

      // CREATE THE ADVISOR SERVICE
      if ($userObj->user_type == 1) {
        $this->createAdvisorServices(ADVISOR_SERVICES, $userObj->id);
      }

      // SEND OTP MAIL
      $mailData = [
        'name' => $userObj->name ?? $userObj->nick_name,
        'email' => $userObj->email,
        'otp' => $otp
      ];

      Mail::to($userObj)->send(new RegisterOtp($mailData));

      $response['data'] = User::where('id', $userObj->id)
        ->with('advisor_category.category_detail:id,title')
        ->with('service_prices')
        ->with('review_rating')
        ->first();

      $response['access_token'] = $token;
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      DB::rollback();

      unset($response['data']);

      $response['message'] = $e->getMessage();

      Log::error($e->getTraceAsString());

      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function login(LoginRequest $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $requestData = $request->all();

      $loginData = User::login($request->all());

      if (!isset($loginData['user'])) {
        $response['message'] = $loginData['message'];
        $response['status'] = STATUS_UNAUTHORIZED;

        return $response;
      }

      $userObj = $loginData['user'];

      if ($requestData['user_type'] != $userObj->user_type) {
        return error_response('Invalid credentials', STATUS_UNAUTHORIZED);
      }

      $token = $userObj->createToken($userObj->id . ' token ')->accessToken;

      $checkActiveService = AdvisorServicePrice::where([
        'advisor_id' => $userObj->id,
        'status' => 1
      ])->first();

      if (!$checkActiveService) {
        $isActive = 0;
      } else {
        $isActive = 1;
      }

      User::where('id', $userObj->id)->update([
        'is_active' => $isActive,
        'device_type' => $requestData['device_type'] ?? NULL,
        'device_token' => $requestData['device_token'] ?? NULL,
        'access_token' => $token,
      ]);

      if ($userObj->user_type == 2 && is_null($userObj->stripe_customer_id)) {
        // CREATE STRIPE CUSTOMER ID
        create_save_stripe_customer_id(StripePaymentGateway::class, $userObj, [
          'email' => $userObj->email
        ]);
      }

      // SEND PUSH NOTIFICATION
      if (!empty($userObj->device_token) && $requestData['device_token'] != $userObj->device_token) {
        $notificationData = [];
        $notificationData['message'] = "Session Expired";
        $notificationData['device_type'] = $userObj->device_type;
        $notificationData['device_token'] = $userObj->device_token;
        $notificationData['notification_type'] = "session_expired";
        $notificationData['details'] = NULL;
        $notificationRes = send_fcm_push_notification($notificationData);

        // SAVE NOTIFICATION
        /* $notificationObj = new Notification;
        $notificationObj->user_id = $userObj->user_id;
        $notificationObj->title = "Order Reply";
        $notificationObj->description = "Session Expired";
        $notificationObj->type = "reply";
        $notificationObj->save(); */
      }

      if ($userObj->user_type == 1) {
        AdvisorServicePrice::where('advisor_id', $userObj->id)->update([
          'status' => 0
        ]);
      }

      // SAVE LOGIN LOGS
      LoginLog::create([
        'user_id' => $userObj->id,
        'type' => 'login'
      ]);

      $response['message'] = "Login successfully";
      $response['data'] = User::where('id', $userObj->id)
        ->with('advisor_category.category_detail:id,title')
        ->with('service_prices')
        ->with('review_rating')
        ->first();
      $response['access_token'] = $token;
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function updatePassword(UpdatePasswordRequest $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $requestData = $request->all();

      $userId = $request->user()->id;

      $userObj = User::find($userId);

      if (!Hash::check($request->old_password, auth()->user()->password)) {
        return error_response("Old Password Doesn't match!", STATUS_BAD_REQUEST);
      }

      $userObj->password = bcrypt($requestData['new_password']);
      $userObj->save();

      $response['message'] = "Password updated successfully";
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function profileDetail(Request $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $requestData = $request->all();

      $userId = $request->user()->id;

      $userObj = User::where('id', $userId)
        ->with('advisor_category.category_detail:id,title')
        ->with('service_prices')
        ->with('review_rating', function ($query) {
          $query->latest()->limit(1);
        })
        ->with('review_rating.user_detail:id,name,nick_name,email,image')
        ->with('review_rating.advisor_detail:id,name,nick_name,email,image')
        ->first();

      $userObj->device_type = $requestData['device_type'] ?? $userObj->device_type;
      $userObj->device_token = $requestData['device_token'] ?? $userObj->device_token;
      $userObj->save();

      $response['message'] = "Profile detail fetched successfully";
      $response['data'] = $userObj;
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function socialLogin(Request $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $post_data = $request->all();

      $post_data['name'] = $post_data['name'] ?? NULL;

      //$rules['email'] = 'required';
      //$rules['name'] = 'required';
      //$rules['username'] = 'required';
      $rules['login_type'] = 'required';

      if (is_null($request->get('login_type'))) {
        $response['errors'] = "Login type is required";
        $response['success'] = FALSE;
        return $response;
      }

      if ($request->get('login_type') == 1) {
        $rules['facebook_id'] = 'required';

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
          $response['message'] = $validator->errors()->first();
          return $response;
        }

        $check_email_exist = User::where('email', $post_data['email'])->first();

        if ($check_email_exist) {
          $userobj = User::where('email', $post_data['email'])->first();
          $check_facebook_id_exist = User::where('email', $post_data['email'])->where('facebook_id', $post_data['facebook_id'])->first();

          $userobj->name = $post_data['name'] ?? $userobj->name;

          if (!$check_facebook_id_exist) {
            $userobj->facebook_id = $post_data['facebook_id'];

            if (isset($post_data['username'])) {
              if ($post_data['username'] != NULL && $post_data['username'] != '') {
                $userobj->username = $post_data['username'];
              }
            }
          }

          $userobj->save();
        } else {
          $userobj = new User;
          $userobj->facebook_id = $post_data['facebook_id'];
          $userobj->name = $post_data['name'];
          $userobj->email = $post_data['email'];
          if (isset($post_data['username'])) {
            if ($post_data['username'] != NULL && $post_data['username'] != '') {
              $userobj->username = $post_data['username'];
            }
          }
          $userobj->password = bcrypt(rand(999999, 6));
          $userobj->save();
        }
      } elseif ($request->get('login_type') == 2) {
        $rules['google_id'] = 'required';

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
          $response['message'] = $validator->errors()->first();
          return $response;
        }

        $check_email_exist = User::where('email', $post_data['email'])->first();

        if ($check_email_exist) {
          $userobj = User::where('email', $post_data['email'])->first();
          $check_google_id_exist = User::where('email', $post_data['email'])->where('google_id', $post_data['google_id'])->first();

          $userobj->name = $post_data['name'] ?? $userobj->name;

          if (!$check_google_id_exist) {
            $userobj->google_id = $post_data['google_id'];

            if (isset($post_data['username'])) {
              if ($post_data['username'] != NULL && $post_data['username'] != '') {
                $userobj->username = $post_data['username'];
              }
            }
          }

          $userobj->save();
        } else {
          $userobj = new User;
          $userobj->google_id = $post_data['google_id'];
          $userobj->name = $post_data['name'];
          $userobj->email = $post_data['email'];
          if (isset($post_data['username'])) {
            if ($post_data['username'] != NULL && $post_data['username'] != '') {
              $userobj->username = $post_data['username'];
            }
          }
          $userobj->password = bcrypt(rand(999999, 6));
          $userobj->save();
        }
      } elseif ($request->get('login_type') == 3) {
        $rules['apple_id'] = 'required';

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
          $response['message'] = $validator->errors()->first();
          return $response;
        }

        $post_data['email'] = $post_data['apple_id'] . "@apple.com";

        $check_email_exist = User::where('email', $post_data['email'])
          ->orWhere('apple_id', $post_data['apple_id'])
          ->first();

        if ($check_email_exist) {
          $userobj = User::where('email', $post_data['email'])->first();
          $check_apple_id_exist = User::where('email', $post_data['email'])->where('apple_id', $post_data['apple_id'])->first();

          $userobj->name = $post_data['name'] ?? $userobj->name;

          if (!$check_apple_id_exist) {
            $userobj->apple_id = $post_data['apple_id'];

            if (isset($post_data['username'])) {
              if ($post_data['username'] != NULL && $post_data['username'] != '') {
                $userobj->username = $post_data['username'];
              }
            }
          }

          $userobj->save();
        } else {
          $userobj = new User;
          $userobj->apple_id = $post_data['apple_id'];
          $userobj->name = $post_data['name'];
          $userobj->email = $post_data['email'];
          if (isset($post_data['username'])) {
            if ($post_data['username'] != NULL && $post_data['username'] != '') {
              $userobj->username = $post_data['username'];
            }
          }
          $userobj->password = bcrypt(rand(999999, 6));
          $userobj->save();
        }
      } elseif ($request->get('login_type') == 4) {
        $rules['instagram_id'] = 'required';

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
          $response['message'] = $validator->errors()->first();
          return $response;
        }

        $check_email_exist = User::where('email', $post_data['email'])->first();

        if ($check_email_exist) {
          $userobj = User::where('email', $post_data['email'])->first();
          $check_instagram_id_exist = User::where('email', $post_data['email'])->where('instagram_id', $post_data['instagram_id'])->first();

          if (!$check_instagram_id_exist) {
            $userobj->instagram_id = $post_data['instagram_id'];
            $userobj->name = $post_data['name'];
            if (isset($post_data['username'])) {
              if ($post_data['username'] != NULL && $post_data['username'] != '') {
                $userobj->username = $post_data['username'];
              }
            }

            $userobj->save();
          }
        } else {
          $userobj = new User;
          $userobj->instagram_id = $post_data['instagram_id'];
          $userobj->name = $post_data['name'];
          $userobj->email = $post_data['email'];
          if (isset($post_data['username'])) {
            if ($post_data['username'] != NULL && $post_data['username'] != '') {
              $userobj->username = $post_data['username'];
            }
          }
          $userobj->password = bcrypt(rand(999999, 6));
          $userobj->save();
        }
      }

      $userobj->is_otp_verified = true;
      $userobj->user_type = 2;
      $userobj->device_type = $post_data['device_type'] ?? NULL;
      $userobj->device_token = $post_data['device_token'] ?? NULL;
      $userobj->save();

      // SAVE 3 FREE MINUTES IF USER TYPE IS 2 (CUSTOMER)
      if ($userobj->user_type == 2 && is_null($userobj->free_minutes)) {
        $userobj->free_minutes = "3:00";
        $userobj->save();
      }

      $user_id = $userobj->id;
      $token = $userobj->createToken($user_id . ' token ')->accessToken;

      $userobj->access_token = $token;
      $userobj->save();

      $response['access_token'] = $token;
      $response['data'] = User::where('id', $user_id)
        ->with('advisor_category.category_detail:id,title')
        ->with('service_prices')
        ->with('review_rating', function ($query) {
          $query->latest()->limit(1);
        })
        ->first();

      // SAVE LOGIN LOGS
      LoginLog::create([
        'user_id' => $user_id,
        'is_social_login' => 1,
        'type' => 'login'
      ]);

      $response['status'] = STATUS_OK;
      $response['success'] = TRUE;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['success'] = FALSE;
    }

    return response()->json($response, $response['status']);
  }

  public function logout(Request $request)
  {
    $response = [];
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      if (Auth::check()) {
        $userId = $request->user()->id;

        User::where('id', $request->user()->id)->update([
          'is_active' => '0',
          'device_type' => NULL,
          'device_token' => NULL,
          'access_token' => NULL,
        ]);

        AdvisorServicePrice::where('advisor_id', $request->user()->id)->update([
          'status' => 0
        ]);

        Session::flush();

        $request->user()->token()->revoke();

        // SAVE LOGIN LOGS
        LoginLog::create([
          'user_id' => $userId,
          'type' => 'logout'
        ]);
      }

      $response['message'] = "Logout successfully";
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      $response['success'] = FALSE;
      $response['status'] = STATUS_GENERAL_ERROR;
      Log::error($e->getTraceAsString());
    }
    return $response;
  }

  public function forgotPassword(ForgotPasswordRequest $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $requestData = $request->all();

      $userObj = User::where('email', $requestData['email'])->first();

      if (!$userObj) {
        return error_response("Email doesn't exist", STATUS_BAD_REQUEST);
      }

      $randomPass = generate_random_string(8);

      $userObj->password = bcrypt($randomPass);
      $userObj->is_one_time_password = true;
      $userObj->save();

      $mail = Mail::to($userObj->email)->send(new ForgotPassword($userObj->name, $randomPass));

      $response['message'] = 'Please check your email for new password';
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      $response['success'] = FALSE;
      Log::error($e->getTraceAsString());
    }

    //$response['response'] = $result;
    return $response;
  }

  public function verifyOtp(VerifyOtpRequest $request)
  {
    $response = [];

    try {
      $requestData = $request->all();

      $userId = $request->user()->id;

      $userObj = User::find($userId);

      // CHECK OTP TIME NOT EXPIRED
      $otpExpiredAt = $userObj->otp_expired_at;

      $currentDateTime = date("Y-m-d H:i:s");

      if ($currentDateTime > $otpExpiredAt) {
        return error_response('OTP expired', STATUS_BAD_REQUEST);
      }

      if ($requestData['otp'] != $userObj->otp) {
        return error_response('Incorrect OTP', STATUS_BAD_REQUEST);
      }

      $userObj->otp = null;
      $userObj->otp_expired_at = null;
      $userObj->is_otp_verified = true;
      $userObj->save();

      $response['message'] = "OTP verified successfully";
      $response['success'] = true;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function countryCodeList(Request $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $resourceObj = CountryCode::get();

      if (empty($resourceObj->toArray())) {
        Artisan::call('db:seed', array('--class' => 'CountryCodeSeeder'));
      }

      $resourceObj = CountryCode::get();

      $response['message'] = "Country code fetched successfully";
      $response['data'] = $resourceObj;
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function resendOtp(Request $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      DB::beginTransaction();

      $userId = $request->user()->id;

      $userObj = User::find($userId);

      // GENERATE OTP
      $otp = generate_otp(6);
      $userObj->otp = $otp;
      $userObj->otp_expired_at = date("Y-m-d H:i:s", strtotime("+10 minutes"));
      $userObj->save();
      DB::commit();

      // SEND OTP MAIL
      $mailData = [
        'name' => $userObj->name ?? $userObj->nick_name,
        'email' => $userObj->email,
        'otp' => $otp
      ];

      Mail::to($userObj)->send(new RegisterOtp($mailData));
      $response['message'] = 'OTP sent successfully';
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      DB::rollback();

      $response['message'] = $e->getMessage();

      Log::error($e->getTraceAsString());

      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function updateProfile(UpdateProfileRequest $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $requestData = $request->all();

      $userId = $request->user()->id;

      $userObj = User::find($userId);

      if (isset($requestData['name']) && !empty($requestData['name'])) {
        $userObj->name = $requestData['name'];
      }

      $userObj->name = $requestData['name'] ?? $userObj->name;
      $userObj->nick_name = $requestData['nick_name'] ?? $userObj->nick_name;
      $userObj->mobile = $requestData['mobile'] ?? $userObj->mobile;
      $userObj->email = $requestData['email'] ?? $userObj->email;
      $userObj->paypal_email = $requestData['paypal_email'] ?? $userObj->paypal_email;
      $userObj->description = $requestData['description'] ?? $userObj->description;

      if (isset($requestData['is_available'])) {
        $userObj->is_available = $requestData['is_available'];
      }

      if ($request->hasFile('image')) {
        if (!is_dir(IMAGE_UPLOAD_PATH)) {
          mkdir(IMAGE_UPLOAD_PATH, 0777, true);
        }

        $file = $request->file('image');
        $fileName = time() . '-' . $file->getClientOriginalName();
        $file->move(IMAGE_UPLOAD_PATH, $fileName);
        $userObj->image = $fileName;
      }

      if (isset($requestData['category']) && !empty($requestData['category'])) {
        $previousCategory = AdvisorCategory::where('advisor_id', $userId)->pluck('category_id')->toArray();
        $categories = $requestData['category'];

        $deletedCategoryIds = array_diff($previousCategory, $categories);

        if (!empty($deletedCategoryIds)) {
          AdvisorCategory::whereIn('category_id', $deletedCategoryIds)->where('advisor_id', $userId)->delete();
        }

        foreach ($categories as $category) {
          $advisorCategoryObj = AdvisorCategory::where([
            'advisor_id' => $userId,
            'category_id' => $category
          ])->first();

          if (!$advisorCategoryObj) {
            $advisorCategoryObj = new AdvisorCategory;
          }

          $advisorCategoryObj->advisor_id = $userId;
          $advisorCategoryObj->category_id = $category;
          $advisorCategoryObj->save();
        }

        $userObj->category_updated_at = date("Y-m-d H:i:s");
      }

      if (isset($requestData['service_price']) && !empty($requestData['service_price'])) {
        $this->setServicePrice($requestData['service_price'], $userId);
      }

      $userObj->save();

      $response['message'] = "Profile updated successfully";
      $response['data'] = User::where('id', $userId)
        ->with('advisor_category.category_detail:id,title')
        ->with('service_prices')
        ->with('review_rating', function ($query) {
          $query->latest()->limit(1);
        })
        ->first();

      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function setServicePrice($servicePrices = [], $userId)
  {
    foreach ($servicePrices as $service => $price) {
      $advisorChatServicePriceObj = AdvisorServicePrice::where([
        'advisor_id' => $userId,
        'service_type' => $service
      ])->first();

      if (!$advisorChatServicePriceObj) {
        $advisorChatServicePriceObj = new AdvisorServicePrice;
        $advisorChatServicePriceObj->advisor_id = $userId;
        $advisorChatServicePriceObj->service_type = $service;
      }

      $advisorChatServicePriceObj->price = $price;
      $advisorChatServicePriceObj->save();
    }
  }

  public function createAdvisorServices($services = [], $userId)
  {
    foreach ($services as $service => $price) {
      $advisorChatServicePriceObj = AdvisorServicePrice::where([
        'advisor_id' => $userId,
        'service_type' => $service
      ])->first();

      if (!$advisorChatServicePriceObj) {
        $advisorChatServicePriceObj = new AdvisorServicePrice;
        $advisorChatServicePriceObj->advisor_id = $userId;
        $advisorChatServicePriceObj->service_type = $service;
      }

      $advisorChatServicePriceObj->price = $price;
      $advisorChatServicePriceObj->save();
    }
  }

  public function updateAvailability(UpdateProfileRequest $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $requestData = $request->all();

      $userId = $request->user()->id;

      $advisorObj = User::select('id', 'name', 'nick_name')->where('id', $userId)->first();

      $services = $requestData;

      $notifiedCustomerIds = FavoriteAdvisor::where('advisor_id', $userId)->pluck('user_id')->toArray();

      $customers = User::select('id', 'device_type', 'device_token')->whereIn('id', $notifiedCustomerIds)->get();

      //print_pre($customers->toArray());

      $isOneServiceActive = AdvisorServicePrice::where([
        'advisor_id' => $userId,
        'status' => 1
      ])->first();

      if (!empty($services)) {
        foreach ($services as $service => $status) {
          $advisorChatServicePriceObj = AdvisorServicePrice::where([
            'advisor_id' => $userId,
            'service_type' => $service
          ])->first();

          if (!$advisorChatServicePriceObj) {
            $advisorChatServicePriceObj = new AdvisorServicePrice;
            $advisorChatServicePriceObj->advisor_id = $userId;
            $advisorChatServicePriceObj->service_type = $service;
            $advisorChatServicePriceObj->price = 0.00;
            $advisorChatServicePriceObj->status = 1;
          }

          $previousAvailabilityStatus = $advisorChatServicePriceObj->status;
          $newAvailabilityStatus = $status;

          $advisorChatServicePriceObj->status = $status;

          $advisorChatServicePriceObj->save();

          // NOTIFICATION WILL ONLY SEND IN THE CASE IF NONE OF THE SERIVCE IS ACTIVE, IF ANY ONE SERVICE IS ALREADY ACTIVE THEN NOTIFICATION WILL NOT SEND
          if (!$isOneServiceActive && $previousAvailabilityStatus == 0 && $newAvailabilityStatus == 1) {
            //die("SD");

            foreach ($customers as $customer) {
              $notificationType = 'update_availability';
              $notificationMessage = $advisorObj->nick_name . " is available for " . (ADVISOR_SERVICES_TEXT[$service] ?? '') . " service";

              $notificationData = [];
              $notificationData['message'] = $notificationMessage;
              $notificationData['device_type'] = $customer->device_type;
              $notificationData['device_token'] = $customer->device_token;
              $notificationData['notification_type'] = $notificationType;
              $notificationData['details'] = [
                'advisor_id' => $userId
              ];
              $notificationRes = send_fcm_push_notification($notificationData);

              // SAVE NOTIFICATION
              $notificationObj = new Notification;
              $notificationObj->user_id = $customer->id;
              $notificationObj->sender_id = $userId;
              $notificationObj->title = "Advisor Available";
              $notificationObj->description = $notificationMessage;
              $notificationObj->type = $notificationType;
              $notificationObj->save();
            }
          }
        }
      }

      // IF NOT ANY SERVICE IS ACTIVE OF ADVISOR THEN UPDATE is_active TO 0 IN users TABLE
      $checkActiveService = AdvisorServicePrice::where([
        'advisor_id' => $userId,
        'status' => 1
      ])->first();

      if (!$checkActiveService) {
        $isActive = 0;
      } else {
        $isActive = 1;
      }

      User::where('id', $userId)->update([
        'is_active' => $isActive
      ]);

      $response['message'] = "Availability updated successfully";

      $response['data'] = User::where('id', $userId)
        ->with('advisor_category.category_detail:id,title')
        ->with('service_prices')
        ->first();
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function activateUser(Request $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $requestData = $request->all();

      $email = $requestData['email'] ?? null;
      $userId = $requestData['user_id'] ?? null;

      $userObj = null;

      if ($email) {
        $userObj = User::where('email', $email)->first();
      }

      if (!$userObj) {
        $userObj = User::find($userId);
      }

      if (!$userObj) {
        return error_response('Invalid request');
      }

      $userObj->is_otp_verified = true;
      $userObj->is_account_verified = true;
      $userObj->save();

      $response['message'] = "Account activated successfully";
      $response['data'] = $userObj;
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function deleteAccount(Request $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $requestData = $request->all();

      User::where('id', $request->user()->id)->delete();

      $response['message'] = "Account deleted successfully";
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function sendFcmPushNotification(Request $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $deviceToken = "e6sq3uSP5Eplis36cPE1I_:APA91bEmX56Uy1iGny8tcZ2asIRybVQmlKS0mXQTN3Mpwm_N1Z4-1gclHfsKCAEDjAx6vuqQ2K8CkW3AM5RRQgPUsd-7i1VETRUzVKBxpe9oj9NS3iT8VhU9Z49Gh8dwbfjbFo0CXntx";

      $notificationData = [];
      $notificationData['message'] = "Received new booking request";
      $notificationData['device_type'] = 1;
      $notificationData['device_token'] = $deviceToken;
      $notificationData['notification_type'] = "24_chat";
      $notificationData['details'] = BookingRequest::where('id', 5)
        ->with('payment_details')
        ->with('booking_request24_detail')
        ->with('user_detail', function ($query) {
          $query->select('id', 'name', 'nick_name', 'email', 'image');
        })
        ->with('advisor_detail', function ($query) {
          $query->select('id', 'name', 'nick_name', 'email', 'image');
        })
        ->first();

      $notificationRes = send_fcm_push_notification($notificationData);

      $response['data'] = $notificationRes;
      $response['message'] = "Notification sent successfully successfully";

      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function sendApnsPushNotification(Request $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $deviceToken = "116db7f3760ef537a2d5a84027587c32484e8522f5ae891360b5c9e4def7f629";

      $notificationData = [
        "body" => [
          "aps" => [
            "alert" => "Test Call",
            "sound" => "default",
            "body" => "Test"
          ],
          "twi_call_sid" => "Test",
          "twi_bridge_token" => "Test",
          "twi_message_type" => "Test",
          "twi_account_sid" => "Test",
          "twi_from" => "Test",
          "twi_to" => "Test",
          "user_id" => "Test",
          "advisor_id" => "Test",
          "booking_id" => "Test",
          "user_name" => "Test",
          "minute" => "Test",
          "receiver_token" => "Test"
        ]
      ];

      $notificationData['device_token'] = $deviceToken;

      /* $notificationRes = send_apns_push_notification($notificationData);
      $response['data'] = $notificationRes; */

      $response['message'] = "Notification sent successfully successfully";

      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }
  public function getNotifications(Request $request)
  {
    $response = [];
    $response['success'] = FALSE;
    $response['status'] = STATUS_BAD_REQUEST;

    try {
      $requestData = $request->all();

      $userId = $request->user()->id;

      $notificationObj = Notification::where('user_id', $userId)->paginate(10);

      $response['message'] = "Notification fetched successfully";
      $response = array_merge($response, $notificationObj->toArray());
      $response['success'] = TRUE;
      $response['status'] = STATUS_OK;
    } catch (Exception $e) {
      $response['message'] = $e->getMessage() . ' Line No ' . $e->getLine() . ' in File' . $e->getFile();
      Log::error($e->getTraceAsString());
      $response['status'] = STATUS_GENERAL_ERROR;
    }

    return response()->json($response, $response['status']);
  }

  public function correctImagePath()
  {
    $users = User::get();

    foreach ($users as $user) {
      $profilePic = $user->image;

      if (!is_null($profilePic)) {
        $imagePathArr = explode("/", $profilePic);

        $imagePath = $profilePic;

        if (count($imagePathArr) > 1) {
          $imagePath = end($imagePathArr);
        }

        User::where('id', $user->id)->update([
          'image' => $imagePath
        ]);
      }
    }
  }
}